"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const signatureTransfer_1 = require("./signatureTransfer");
const constants_1 = require("./constants");
describe('SignatureTransfer', () => {
    describe('Max values', () => {
        it('max values', () => {
            expect(() => signatureTransfer_1.SignatureTransfer.hash({
                permitted: {
                    token: '0x0000000000000000000000000000000000000000',
                    amount: constants_1.MaxSignatureTransferAmount.toString(),
                },
                spender: '0x0000000000000000000000000000000000000000',
                nonce: constants_1.MaxUnorderedNonce.toString(),
                deadline: constants_1.MaxSigDeadline.toString(),
            }, '0x0000000000000000000000000000000000000000', 1)).not.toThrow();
        });
        it('nonce out of range', () => {
            expect(() => signatureTransfer_1.SignatureTransfer.hash({
                permitted: {
                    token: '0x0000000000000000000000000000000000000000',
                    amount: '0',
                },
                spender: '0x0000000000000000000000000000000000000000',
                nonce: constants_1.MaxUnorderedNonce.add(1).toString(),
                deadline: '0',
            }, '0x0000000000000000000000000000000000000000', 1)).toThrow('NONCE_OUT_OF_RANGE');
        });
        it('amount out of range', () => {
            expect(() => signatureTransfer_1.SignatureTransfer.hash({
                permitted: {
                    token: '0x0000000000000000000000000000000000000000',
                    amount: constants_1.MaxSignatureTransferAmount.add(1).toString(),
                },
                spender: '0x0000000000000000000000000000000000000000',
                nonce: '0',
                deadline: '0',
            }, '0x0000000000000000000000000000000000000000', 1)).toThrow('AMOUNT_OUT_OF_RANGE');
        });
        it('deadline out of range', () => {
            expect(() => signatureTransfer_1.SignatureTransfer.hash({
                permitted: {
                    token: '0x0000000000000000000000000000000000000000',
                    amount: '0',
                },
                spender: '0x0000000000000000000000000000000000000000',
                nonce: '0',
                deadline: constants_1.MaxSigDeadline.add(1).toString(),
            }, '0x0000000000000000000000000000000000000000', 1)).toThrow('SIG_DEADLINE_OUT_OF_RANGE');
        });
    });
    it('non-batch, no witness', () => {
        expect(signatureTransfer_1.SignatureTransfer.hash({
            permitted: {
                token: '0x0000000000000000000000000000000000000000',
                amount: '0',
            },
            spender: '0x0000000000000000000000000000000000000000',
            nonce: '0',
            deadline: '0',
        }, '0x0000000000000000000000000000000000000000', 1)).toBe('0xb9bf9813799d7f0de28d2142b0bc80ec289d4a6a5b9f41834149df4188804dc5');
    });
    it('non-batch, witness', () => {
        expect(signatureTransfer_1.SignatureTransfer.hash({
            permitted: {
                token: '0x0000000000000000000000000000000000000000',
                amount: '0',
            },
            spender: '0x0000000000000000000000000000000000000000',
            nonce: '0',
            deadline: '0',
        }, '0x0000000000000000000000000000000000000000', 1, {
            witnessTypeName: 'MockWitness',
            witnessType: { MockWitness: [{ name: 'mock', type: 'uint256' }] },
            witness: { mock: '0x0000000000000000000000000000000000000000000000000000000000000000' },
        })).toBe('0x4236a4a7b3e8e65dbb4cc758ef10dc4887e2959853fb615140d0f5e0ae7be7c9');
    });
    it('batch, no witness', () => {
        expect(signatureTransfer_1.SignatureTransfer.hash({
            permitted: [
                {
                    token: '0x0000000000000000000000000000000000000000',
                    amount: '0',
                },
            ],
            spender: '0x0000000000000000000000000000000000000000',
            nonce: '0',
            deadline: '0',
        }, '0x0000000000000000000000000000000000000000', 1)).toBe('0x5ba40c5ba725fec181e4a862c717adf91682b012ad01ea99a978189106d66923');
    });
    it('batch, witness', () => {
        expect(signatureTransfer_1.SignatureTransfer.hash({
            permitted: [
                {
                    token: '0x0000000000000000000000000000000000000000',
                    amount: '0',
                },
            ],
            spender: '0x0000000000000000000000000000000000000000',
            nonce: '0',
            deadline: '0',
        }, '0x0000000000000000000000000000000000000000', 1, {
            witnessTypeName: 'MockWitness',
            witnessType: { MockWitness: [{ name: 'mock', type: 'uint256' }] },
            witness: { mock: '0x0000000000000000000000000000000000000000000000000000000000000000' },
        })).toBe('0xb45d605b0a4d4f16930a4f48294d94c78f34411278fd3133626cc190273e3ccf');
    });
});
//# sourceMappingURL=signatureTransfer.test.js.map